
------------------------------------------------------------------------------
-- Title       : sdramc.vhd
-- Description : SDRAMRg[
-- Author      : KAWAMOTO Yasuhisa
-- Mail        : kawamoto@devdrv.co.jp
-- Date        : 10/22/2007
------------------------------------------------------------------------------

library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;

entity SDRAMC is
	generic (
		BUFLEN    : integer := 32
	);
	port (
		RST       : in std_logic;
		CLK       : in std_logic;

		CPUADDR     : in std_logic_vector(23 downto 0);
		CPUWDATA    : in std_logic_vector(7 downto 0);
		CPUWDATASTB : in std_logic;
		CPUWDATABSY : out std_logic;
		CPURDATA    : out std_logic_vector(7 downto 0);
		CPURDATASTB : in std_logic;
		CPURDATABSY : out std_logic;

		CRTBUFFSH : in std_logic;
		CRTBUFCLR : in std_logic;
		CRTBUFWD  : out std_logic_vector(31 downto 0);
		CRTBUFADR : out integer range 0 to BUFLEN - 1;
		CRTBUFWE  : out std_logic;

		CAMBUFFSH : in std_logic;
		CAMBUFCLR : in std_logic;
		CAMBUFRD  : in std_logic_vector(31 downto 0);
		CAMBUFADR : out integer range 0 to BUFLEN - 1;
		CAMSTART  : in std_logic;
		CAMBUSY   : out std_logic;

		SD_LOCK   : in std_logic;
		SD_CKE    : out std_logic;
		SD_CS     : out std_logic;
		SD_RAS    : out std_logic;
		SD_CAS    : out std_logic;
		SD_WE     : out std_logic;
		SD_DQM    : out std_logic_vector( 3 downto 0);
		SD_MBA    : out std_logic_vector( 1 downto 0);
		SD_MA     : out std_logic_vector(11 downto 0);
		SD_MD_i   : in std_logic_vector(31 downto 0);
		SD_MD_o   : out std_logic_vector(31 downto 0);
		SD_MD_t   : out std_logic;
		
		DBG_OUT   : out std_logic_vector(3 downto 0)
	);
end SDRAMC;

architecture RTL of SDRAMC is

	type DEF_STATE is (IDLE, INIT, STBY, CREAD, CWRITE, PREAD, PWRITE, REF);
	signal STATE, NEXT_STATE : DEF_STATE;
	signal COUNT, NEXT_COUNT : integer range 0 to 11;

	signal LOCK_CNT : integer range 0 to 4800;
	signal INITREQ  : std_logic;

	signal REF_CNT : integer range 0 to 750;
	signal REFREQ  : std_logic;

	signal CRTBUFWA : integer range 0 to BUFLEN - 1;
	signal CRTBUFF  : std_logic_vector(1 downto 0);
	signal CRTREQ   : std_logic;
	signal CRTADDR  : std_logic_vector(21 downto 0);

	signal CAMBUFRA : integer range 0 to BUFLEN - 1;
	signal CAMBUFF  : std_logic_vector(1 downto 0);
	signal CAMREQ   : std_logic;
	signal CAMADDR  : std_logic_vector(21 downto 0);

	signal CRTBUFFSH_S : std_logic_vector(3 downto 1);
	signal CRTBUFCLR_S : std_logic_vector(3 downto 1);
	signal CAMBUFFSH_S : std_logic_vector(3 downto 1);
	signal CAMBUFCLR_S : std_logic_vector(3 downto 1);

	signal CAMENB      : std_logic;

	signal PRDREQ   : std_logic;
	signal PWRREQ   : std_logic;

	signal SD_CRD : std_logic;
	signal SD_CRA : std_logic;
	signal SD_CWD : std_logic;
	signal SD_CWA : std_logic;
	signal SD_PRD : std_logic;
	signal SD_PWD : std_logic;
	signal SD_REF : std_logic;

	signal SD_PWD_D : std_logic;

	signal NEXT_SD_CS  : std_logic;
	signal NEXT_SD_RAS : std_logic;
	signal NEXT_SD_CAS : std_logic;
	signal NEXT_SD_WE  : std_logic;
	signal NEXT_SD_DQM : std_logic_vector( 3 downto 0);
	signal NEXT_SD_MBA : std_logic_vector( 1 downto 0);
	signal NEXT_SD_MA  : std_logic_vector(11 downto 0);
	signal NEXT_SD_CRD  : std_logic;
	signal NEXT_SD_CRA  : std_logic;
	signal NEXT_SD_CWD  : std_logic;
	signal NEXT_SD_CWA  : std_logic;
	signal NEXT_SD_PRD  : std_logic;
	signal NEXT_SD_PWD  : std_logic;
	signal NEXT_SD_REF : std_logic;
begin

-- CKE(100usEFCg)
	process (CLK, RST) begin
		if (RST = '1') then
			INITREQ  <= '0';
			LOCK_CNT <= 0;
		elsif (CLK'event and CLK = '1') then
--			if (SD_LOCK = '1') then
				if (LOCK_CNT = 4800) then		-- 100us(@48MHz)
					INITREQ <= '1';
				else
					LOCK_CNT <= LOCK_CNT + 1;
				end if;
--			end if;
		end if;
	end process;

-- tbV
	process (CLK, RST) begin
		if (RST = '1') then
			REF_CNT <= 0;
			REFREQ  <= '0';
		elsif (CLK'event and CLK = '1') then
			REF_CNT <= REF_CNT + 1;
			if (REF_CNT = 750) then				-- 64ms/4096(@48MHz)
				REF_CNT <= 0;
				REFREQ <= '1';
			elsif (SD_REF = '1') then
				REFREQ <= '0';
			end if;
		end if;
	end process;

-- 
	process (CLK) begin
		if (CLK'event and CLK = '1') then
			CRTBUFFSH_S(1) <= CRTBUFFSH;
			CRTBUFFSH_S(2) <= CRTBUFFSH_S(1);
			CRTBUFFSH_S(3) <= CRTBUFFSH_S(2);
		end if;
	end process;

	process (CLK) begin
		if (CLK'event and CLK = '1') then
			CRTBUFCLR_S(1) <= CRTBUFCLR;
			CRTBUFCLR_S(2) <= CRTBUFCLR_S(1);
			CRTBUFCLR_S(3) <= CRTBUFCLR_S(2);
		end if;
	end process;

	process (CLK) begin
		if (CLK'event and CLK = '1') then
			CAMBUFFSH_S(1) <= CAMBUFFSH;
			CAMBUFFSH_S(2) <= CAMBUFFSH_S(1);
			CAMBUFFSH_S(3) <= CAMBUFFSH_S(2);
		end if;
	end process;

	process (CLK) begin
		if (CLK'event and CLK = '1') then
			CAMBUFCLR_S(1) <= CAMBUFCLR;
			CAMBUFCLR_S(2) <= CAMBUFCLR_S(1);
			CAMBUFCLR_S(3) <= CAMBUFCLR_S(2);
		end if;
	end process;

-- \pobt@ECg
	process (CLK, RST) begin
		if (RST = '1') then
			CRTBUFWA <= 0;
			CRTBUFF  <= (others => '0');
			CRTADDR  <= (others => '0');
		elsif (CLK'event and CLK = '1') then
			if (CRTBUFFSH_S(2) = '1' and CRTBUFFSH_S(3) = '0') then
				CRTBUFWA <= 0;
				CRTBUFF  <= (others => '0');
				CRTADDR  <= (others => '0');
			else
				if (SD_CRD = '1') then
					if (CRTBUFWA = BUFLEN / 2 - 1) then
						CRTBUFF(0) <= '1';
						CRTBUFWA   <= CRTBUFWA + 1;
					elsif (CRTBUFWA = BUFLEN - 1) then
						CRTBUFF(1) <= '1';
						CRTBUFWA   <= 0;
					else
						CRTBUFWA   <= CRTBUFWA + 1;
					end if;
				end if;
				
				if (SD_CRA = '1') then
					CRTADDR <= CRTADDR + 1;
				end if;
				
				if (CRTBUFCLR_S(2) = '1' and CRTBUFCLR_S(3) = '0') then
					CRTBUFF(0) <= '0';
				end if;
				
				if (CRTBUFCLR_S(2) = '0' and CRTBUFCLR_S(3) = '1') then
					CRTBUFF(1) <= '0';
				end if;
			end if;
		end if;
	end process;

	CRTBUFWD <= SD_MD_i;
	CRTBUFADR <= CRTBUFWA;
	CRTBUFWE <= SD_CRD;
	CRTREQ <= (not CRTBUFF(0)) or (not CRTBUFF(1));

-- Jpobt@E[h
	process (CLK, RST) begin
		if (RST = '1') then
			CAMBUFRA <= 0;
			CAMBUFF  <= (others => '0');
			CAMADDR  <= (others => '0');
		elsif (CLK'event and CLK = '1') then
			if (CAMBUFFSH_S(2) = '1' and CAMBUFFSH_S(3) = '0') then
				CAMBUFRA <= 0;
				CAMBUFF  <= (others => '0');
				CAMADDR  <= (others => '0');
			else
				if (SD_CWD = '1') then
					if (CAMBUFRA = BUFLEN / 2 - 1) then
						CAMBUFF(0) <= '0';
						CAMBUFRA   <= CAMBUFRA + 1;
					elsif (CAMBUFRA = BUFLEN - 1) then
						CAMBUFF(1) <= '0';
						CAMBUFRA   <= 0;
					else
						CAMBUFRA   <= CAMBUFRA + 1;
					end if;
				end if;
				
				if (SD_CWA = '1') then
					CAMADDR <= CAMADDR + 1;
				end if;
				
				if (CAMBUFCLR_S(2) = '1' and CAMBUFCLR_S(3) = '0') then
					CAMBUFF(0) <= '1';
				end if;
				
				if (CAMBUFCLR_S(2) = '0' and CAMBUFCLR_S(3) = '1') then
					CAMBUFF(1) <= '1';
				end if;
			end if;
		end if;
	end process;

	process (CLK, RST) begin
		if (RST = '1') then
			CAMENB <= '0';
		elsif (CLK'event and CLK = '1') then
			if (CAMBUFFSH_S(2) = '1' and CAMBUFFSH_S(3) = '0') then
				if (CAMSTART = '1') then
					CAMENB <= '1';
				else
					CAMENB <= '0';
				end if;
			end if;
		end if;
	end process;

	CAMBUSY   <= CAMENB;
	CAMBUFADR <= CAMBUFRA;
	CAMREQ    <= (CAMBUFF(0) or CAMBUFF(1)) and CAMENB;

-- CPUCg
	process (CLK, RST) begin
		if (RST = '1') then
			PWRREQ <= '0';
		elsif (CLK'event and CLK = '1') then
			if (CPUWDATASTB = '1') then
				PWRREQ <= '1';
			elsif (SD_PWD = '1') then
				PWRREQ <= '0';
			end if;
		end if;
	end process;

	CPUWDATABSY <= PWRREQ;

-- CPU[h
	process (CLK, RST) begin
		if (RST = '1') then
			PRDREQ <= '0';
		elsif (CLK'event and CLK = '1') then
			if (CPURDATASTB = '1') then
				PRDREQ <= '1';
			elsif (SD_PRD = '1') then
				PRDREQ <= '0';
			end if;
		end if;
	end process;

	CPURDATABSY <= PRDREQ;

	process (CLK) begin
		if (CLK'event and CLK = '1') then
			if (SD_PRD = '1') then
				if (CPUADDR(1 downto 0) = "00") then
					CPURDATA <= SD_MD_i( 7 downto  0);
				elsif (CPUADDR(1 downto 0) = "01") then
					CPURDATA <= SD_MD_i(15 downto  8);
				elsif (CPUADDR(1 downto 0) = "10") then
					CPURDATA <= SD_MD_i(23 downto 16);
				else
					CPURDATA <= SD_MD_i(31 downto 24);
				end if;
			end if;
		end if;
	end process;

-- SDRAM
	process (CLK, RST) begin
		if (RST = '1') then
			STATE  <= IDLE;
			COUNT  <= 0;
			SD_CS  <= '0';
			SD_RAS <= '0';
			SD_CAS <= '0';
			SD_WE  <= '0';
			SD_DQM <= "0000";
			SD_MBA <= "00";
			SD_MA  <= "000000000000";
			SD_CRD <= '0';
			SD_CRA <= '0';
			SD_CWD <= '0';
			SD_CWA <= '0';
			SD_PRD <= '0';
			SD_PWD <= '0';
			SD_REF <= '0';
		elsif (CLK'event and CLK = '1') then
			STATE  <= NEXT_STATE;
			COUNT  <= NEXT_COUNT;
			SD_CS  <= NEXT_SD_CS;
			SD_RAS <= NEXT_SD_RAS;
			SD_CAS <= NEXT_SD_CAS;
			SD_WE  <= NEXT_SD_WE;
			SD_DQM <= NEXT_SD_DQM;
			SD_MBA <= NEXT_SD_MBA;
			SD_MA  <= NEXT_SD_MA;
			SD_CRD <= NEXT_SD_CRD;
			SD_CRA <= NEXT_SD_CRA;
			SD_CWD <= NEXT_SD_CWD;
			SD_CWA <= NEXT_SD_CWA;
			SD_PRD <= NEXT_SD_PRD;
			SD_PWD <= NEXT_SD_PWD;
			SD_REF <= NEXT_SD_REF;
		end if;
	end process;
	
	process (STATE, COUNT) begin
		case STATE is
			when IDLE =>					-- INH
				DBG_OUT <= "0000";			-- DEBUG(----)
				NEXT_SD_CS  <= '0';
				NEXT_SD_RAS <= '0';
				NEXT_SD_CAS <= '0';
				NEXT_SD_WE  <= '0';
				NEXT_SD_DQM <= "0000";
				NEXT_SD_MBA <= "00";
				NEXT_SD_MA  <= "000000000000";
				NEXT_SD_CRD <= '0';
				NEXT_SD_CRA <= '0';
				NEXT_SD_CWD <= '0';
				NEXT_SD_CWA <= '0';
				NEXT_SD_PRD <= '0';
				NEXT_SD_PWD <= '0';
				NEXT_SD_REF <= '0';
				if (INITREQ = '1') then
					NEXT_COUNT <= 0;
					NEXT_STATE <= INIT;
				else
					NEXT_COUNT <= COUNT;
					NEXT_STATE <= STATE;
				end if;
			when INIT =>
				DBG_OUT <= "0001";			-- DEBUG(---*)
				case COUNT is
					when 0 =>				-- NOP
						NEXT_SD_CS  <= '1';
						NEXT_SD_RAS <= '0';
						NEXT_SD_CAS <= '0';
						NEXT_SD_WE  <= '0';
						NEXT_SD_DQM <= "0000";
						NEXT_SD_MBA <= "00";
						NEXT_SD_MA  <= "000000000000";
						NEXT_SD_CRD <= '0';
						NEXT_SD_CRA <= '0';
						NEXT_SD_CWD <= '0';
						NEXT_SD_CWA <= '0';
						NEXT_SD_PRD <= '0';
						NEXT_SD_PWD <= '0';
						NEXT_SD_REF <= '0';
						NEXT_COUNT  <= COUNT + 1;
						NEXT_STATE  <= STATE;
					when 1 =>				-- ALL PRE
						NEXT_SD_CS  <= '1';
						NEXT_SD_RAS <= '1';
						NEXT_SD_CAS <= '0';
						NEXT_SD_WE  <= '1';
						NEXT_SD_DQM <= "0000";
						NEXT_SD_MBA <= "00";
						NEXT_SD_MA  <= "010000000000";
						NEXT_SD_CRD <= '0';
						NEXT_SD_CRA <= '0';
						NEXT_SD_CWD <= '0';
						NEXT_SD_CWA <= '0';
						NEXT_SD_PRD <= '0';
						NEXT_SD_PWD <= '0';
						NEXT_SD_REF <= '0';
						NEXT_COUNT  <= COUNT + 1;
						NEXT_STATE  <= STATE;
					when 2 =>				-- REF
						NEXT_SD_CS  <= '1';
						NEXT_SD_RAS <= '1';
						NEXT_SD_CAS <= '1';
						NEXT_SD_WE  <= '0';
						NEXT_SD_DQM <= "0000";
						NEXT_SD_MBA <= "00";
						NEXT_SD_MA  <= "000000000000";
						NEXT_SD_CRD <= '0';
						NEXT_SD_CRA <= '0';
						NEXT_SD_CWD <= '0';
						NEXT_SD_CWA <= '0';
						NEXT_SD_PRD <= '0';
						NEXT_SD_PWD <= '0';
						NEXT_SD_REF <= '0';
						NEXT_COUNT  <= COUNT + 1;
						NEXT_STATE  <= STATE;
					when 3 to 4 =>			-- NOP
						NEXT_SD_CS  <= '1';
						NEXT_SD_RAS <= '0';
						NEXT_SD_CAS <= '0';
						NEXT_SD_WE  <= '0';
						NEXT_SD_DQM <= "0000";
						NEXT_SD_MBA <= "00";
						NEXT_SD_MA  <= "000000000000";
						NEXT_SD_CRD <= '0';
						NEXT_SD_CRA <= '0';
						NEXT_SD_CWD <= '0';
						NEXT_SD_CWA <= '0';
						NEXT_SD_PRD <= '0';
						NEXT_SD_PWD <= '0';
						NEXT_SD_REF <= '0';
						NEXT_COUNT  <= COUNT + 1;
						NEXT_STATE  <= STATE;
					when 5 =>				-- REF
						NEXT_SD_CS  <= '1';
						NEXT_SD_RAS <= '1';
						NEXT_SD_CAS <= '1';
						NEXT_SD_WE  <= '0';
						NEXT_SD_DQM <= "0000";
						NEXT_SD_MBA <= "00";
						NEXT_SD_MA  <= "000000000000";
						NEXT_SD_CRD <= '0';
						NEXT_SD_CRA <= '0';
						NEXT_SD_CWD <= '0';
						NEXT_SD_CWA <= '0';
						NEXT_SD_PRD <= '0';
						NEXT_SD_PWD <= '0';
						NEXT_SD_REF <= '0';
						NEXT_COUNT  <= COUNT + 1;
						NEXT_STATE  <= STATE;
					when 6 to 7 =>			-- NOP
						NEXT_SD_CS  <= '1';
						NEXT_SD_RAS <= '0';
						NEXT_SD_CAS <= '0';
						NEXT_SD_WE  <= '0';
						NEXT_SD_DQM <= "0000";
						NEXT_SD_MBA <= "00";
						NEXT_SD_MA  <= "000000000000";
						NEXT_SD_CRD <= '0';
						NEXT_SD_CRA <= '0';
						NEXT_SD_CWD <= '0';
						NEXT_SD_CWA <= '0';
						NEXT_SD_PRD <= '0';
						NEXT_SD_PWD <= '0';
						NEXT_SD_REF <= '0';
						NEXT_COUNT  <= COUNT + 1;
						NEXT_STATE  <= STATE;
					when others =>			-- LMR
						NEXT_SD_CS  <= '1';
						NEXT_SD_RAS <= '1';
						NEXT_SD_CAS <= '1';
						NEXT_SD_WE  <= '1';
						NEXT_SD_DQM <= "0000";
						NEXT_SD_MBA <= "00";
						NEXT_SD_MA  <= "000000010011";
						NEXT_SD_CRD <= '0';
						NEXT_SD_CRA <= '0';
						NEXT_SD_CWD <= '0';
						NEXT_SD_CWA <= '0';
						NEXT_SD_PRD <= '0';
						NEXT_SD_PWD <= '0';
						NEXT_SD_REF <= '0';
						NEXT_COUNT  <= 0;
						NEXT_STATE  <= STBY;
				end case;
			when STBY =>					-- NOP
				DBG_OUT <= "0000";			-- DEBUG(----)
				NEXT_SD_CS  <= '1';
				NEXT_SD_RAS <= '0';
				NEXT_SD_CAS <= '0';
				NEXT_SD_WE  <= '0';
				NEXT_SD_DQM <= "0000";
				NEXT_SD_MBA <= "00";
				NEXT_SD_MA  <= "000000000000";
				NEXT_SD_CRD <= '0';
				NEXT_SD_CRA <= '0';
				NEXT_SD_CWD <= '0';
				NEXT_SD_CWA <= '0';
				NEXT_SD_PRD <= '0';
				NEXT_SD_PWD <= '0';
				NEXT_SD_REF <= '0';
				if (CRTREQ = '1') then
					NEXT_COUNT <= 0;
					NEXT_STATE <= CREAD;
				elsif (CAMREQ = '1') then
					NEXT_COUNT <= 0;
					NEXT_STATE <= CWRITE;
				elsif (PRDREQ = '1') then
					NEXT_COUNT <= 0;
					NEXT_STATE <= PREAD;
				elsif (PWRREQ = '1') then
					NEXT_COUNT <= 0;
					NEXT_STATE <= PWRITE;
				elsif (REFREQ = '1') then
					NEXT_COUNT <= 0;
					NEXT_STATE <= REF;
				else
					NEXT_COUNT <= COUNT;
					NEXT_STATE <= STATE;
				end if;
			when CREAD =>
				DBG_OUT <= "0010";			-- DEBUG(--*-)
				case COUNT is
					when 0 =>				-- ACT
						NEXT_SD_CS  <= '1';
						NEXT_SD_RAS <= '1';
						NEXT_SD_CAS <= '0';
						NEXT_SD_WE  <= '0';
						NEXT_SD_DQM <= "0000";
						NEXT_SD_MBA <= CRTADDR(21 downto 20);
						NEXT_SD_MA  <= CRTADDR(19 downto  8);
						NEXT_SD_CRD <= '0';
						NEXT_SD_CRA <= '0';
						NEXT_SD_CWD <= '0';
						NEXT_SD_CWA <= '0';
						NEXT_SD_PRD <= '0';
						NEXT_SD_PWD <= '0';
						NEXT_SD_REF <= '0';
						NEXT_COUNT  <= COUNT + 1;
						NEXT_STATE  <= STATE;
					when 1 =>				-- READ(AUTO PRE)
						NEXT_SD_CS  <= '1';
						NEXT_SD_RAS <= '0';
						NEXT_SD_CAS <= '1';
						NEXT_SD_WE  <= '0';
						NEXT_SD_DQM <= "0000";
						NEXT_SD_MBA <= CRTADDR(21 downto 20);
						NEXT_SD_MA  <= "0100" & CRTADDR(7 downto 0);
						NEXT_SD_CRD <= '0';
						NEXT_SD_CRA <= '0';
						NEXT_SD_CWD <= '0';
						NEXT_SD_CWA <= '0';
						NEXT_SD_PRD <= '0';
						NEXT_SD_PWD <= '0';
						NEXT_SD_REF <= '0';
						NEXT_COUNT  <= COUNT + 1;
						NEXT_STATE  <= STATE;
					when 2 to 9 =>			-- NOP
						NEXT_SD_CS  <= '1';
						NEXT_SD_RAS <= '0';
						NEXT_SD_CAS <= '0';
						NEXT_SD_WE  <= '0';
						NEXT_SD_DQM <= "0000";
						NEXT_SD_MBA <= "00";
						NEXT_SD_MA  <= "000000000000";
						NEXT_SD_CRD <= '1';
						NEXT_SD_CRA <= '1';
						NEXT_SD_CWD <= '0';
						NEXT_SD_CWA <= '0';
						NEXT_SD_PRD <= '0';
						NEXT_SD_PWD <= '0';
						NEXT_SD_REF <= '0';
						NEXT_COUNT  <= COUNT + 1;
						NEXT_STATE  <= STATE;
					when others =>			-- NOP
						NEXT_SD_CS  <= '1';
						NEXT_SD_RAS <= '0';
						NEXT_SD_CAS <= '0';
						NEXT_SD_WE  <= '0';
						NEXT_SD_DQM <= "0000";
						NEXT_SD_MBA <= "00";
						NEXT_SD_MA  <= "000000000000";
						NEXT_SD_CRD <= '0';
						NEXT_SD_CRA <= '0';
						NEXT_SD_CWD <= '0';
						NEXT_SD_CWA <= '0';
						NEXT_SD_PRD <= '0';
						NEXT_SD_PWD <= '0';
						NEXT_SD_REF <= '0';
						NEXT_COUNT  <= 0;
						NEXT_STATE  <= STBY;
				end case;
			when CWRITE =>
				DBG_OUT <= "0100";			-- DEBUG(-*--)
				case COUNT is
					when 0 =>				-- ACT
						NEXT_SD_CS  <= '1';
						NEXT_SD_RAS <= '1';
						NEXT_SD_CAS <= '0';
						NEXT_SD_WE  <= '0';
						NEXT_SD_DQM <= "0000";
						NEXT_SD_MBA <= CAMADDR(21 downto 20);
						NEXT_SD_MA  <= CAMADDR(19 downto  8);
						NEXT_SD_CRD <= '0';
						NEXT_SD_CRA <= '0';
						NEXT_SD_CWD <= '1';
						NEXT_SD_CWA <= '0';
						NEXT_SD_PRD <= '0';
						NEXT_SD_PWD <= '0';
						NEXT_SD_REF <= '0';
						NEXT_COUNT  <= COUNT + 1;
						NEXT_STATE  <= STATE;
					when 1 =>				-- WRITE(AUTO PRE)
						NEXT_SD_CS  <= '1';
						NEXT_SD_RAS <= '0';
						NEXT_SD_CAS <= '1';
						NEXT_SD_WE  <= '1';
						NEXT_SD_DQM <= "0000";
						NEXT_SD_MBA <= CAMADDR(21 downto 20);
						NEXT_SD_MA  <= "0100" & CAMADDR(7 downto  0);
						NEXT_SD_CRD <= '0';
						NEXT_SD_CRA <= '0';
						NEXT_SD_CWD <= '1';
						NEXT_SD_CWA <= '1';
						NEXT_SD_PRD <= '0';
						NEXT_SD_PWD <= '0';
						NEXT_SD_REF <= '0';
						NEXT_COUNT  <= COUNT + 1;
						NEXT_STATE  <= STATE;
					when 2 to 7 =>			-- NOP
						NEXT_SD_CS  <= '1';
						NEXT_SD_RAS <= '0';
						NEXT_SD_CAS <= '0';
						NEXT_SD_WE  <= '0';
						NEXT_SD_DQM <= "0000";
						NEXT_SD_MBA <= "00";
						NEXT_SD_MA  <= "000000000000";
						NEXT_SD_CRD <= '0';
						NEXT_SD_CRA <= '0';
						NEXT_SD_CWD <= '1';
						NEXT_SD_CWA <= '1';
						NEXT_SD_PRD <= '0';
						NEXT_SD_PWD <= '0';
						NEXT_SD_REF <= '0';
						NEXT_COUNT  <= COUNT + 1;
						NEXT_STATE  <= STATE;
					when 8 =>				-- NOP
						NEXT_SD_CS  <= '1';
						NEXT_SD_RAS <= '0';
						NEXT_SD_CAS <= '0';
						NEXT_SD_WE  <= '0';
						NEXT_SD_DQM <= "0000";
						NEXT_SD_MBA <= "00";
						NEXT_SD_MA  <= "000000000000";
						NEXT_SD_CRD <= '0';
						NEXT_SD_CRA <= '0';
						NEXT_SD_CWD <= '0';
						NEXT_SD_CWA <= '1';
						NEXT_SD_PRD <= '0';
						NEXT_SD_PWD <= '0';
						NEXT_SD_REF <= '0';
						NEXT_COUNT  <= COUNT + 1;
						NEXT_STATE  <= STATE;
					when 9 =>				-- NOP
						NEXT_SD_CS  <= '1';
						NEXT_SD_RAS <= '0';
						NEXT_SD_CAS <= '0';
						NEXT_SD_WE  <= '0';
						NEXT_SD_DQM <= "0000";
						NEXT_SD_MBA <= "00";
						NEXT_SD_MA  <= "000000000000";
						NEXT_SD_CRD <= '0';
						NEXT_SD_CRA <= '0';
						NEXT_SD_CWD <= '0';
						NEXT_SD_CWA <= '0';
						NEXT_SD_PRD <= '0';
						NEXT_SD_PWD <= '0';
						NEXT_SD_REF <= '0';
						NEXT_COUNT  <= COUNT + 1;
						NEXT_STATE  <= STATE;
					when others =>			-- NOP
						NEXT_SD_CS  <= '1';
						NEXT_SD_RAS <= '0';
						NEXT_SD_CAS <= '0';
						NEXT_SD_WE  <= '0';
						NEXT_SD_DQM <= "0000";
						NEXT_SD_MBA <= "00";
						NEXT_SD_MA  <= "000000000000";
						NEXT_SD_CRD <= '0';
						NEXT_SD_CRA <= '0';
						NEXT_SD_CWD <= '0';
						NEXT_SD_CWA <= '0';
						NEXT_SD_PRD <= '0';
						NEXT_SD_PWD <= '0';
						NEXT_SD_REF <= '0';
						NEXT_COUNT  <= 0;
						NEXT_STATE  <= STBY;
				end case;
			when PREAD =>
				DBG_OUT <= "0010";			-- DEBUG(--*-)
				case COUNT is
					when 0 =>				-- ACT
						NEXT_SD_CS  <= '1';
						NEXT_SD_RAS <= '1';
						NEXT_SD_CAS <= '0';
						NEXT_SD_WE  <= '0';
						NEXT_SD_DQM <= "0000";
						NEXT_SD_MBA <= CPUADDR(23 downto 22);
						NEXT_SD_MA  <= CPUADDR(21 downto  10);
						NEXT_SD_CRD <= '0';
						NEXT_SD_CRA <= '0';
						NEXT_SD_CWD <= '0';
						NEXT_SD_CWA <= '0';
						NEXT_SD_PRD <= '0';
						NEXT_SD_PWD <= '0';
						NEXT_SD_REF <= '0';
						NEXT_COUNT  <= COUNT + 1;
						NEXT_STATE  <= STATE;
					when 1 =>				-- READ(AUTO PRE)
						NEXT_SD_CS  <= '1';
						NEXT_SD_RAS <= '0';
						NEXT_SD_CAS <= '1';
						NEXT_SD_WE  <= '0';
						NEXT_SD_DQM <= "0000";
						NEXT_SD_MBA <= CPUADDR(23 downto 22);
						NEXT_SD_MA  <= "0100" & CPUADDR(9 downto 2);
						NEXT_SD_CRD <= '0';
						NEXT_SD_CRA <= '0';
						NEXT_SD_CWD <= '0';
						NEXT_SD_CWA <= '0';
						NEXT_SD_PRD <= '0';
						NEXT_SD_PWD <= '0';
						NEXT_SD_REF <= '0';
						NEXT_COUNT  <= COUNT + 1;
						NEXT_STATE  <= STATE;
					when 2 =>			-- NOP
						NEXT_SD_CS  <= '1';
						NEXT_SD_RAS <= '0';
						NEXT_SD_CAS <= '0';
						NEXT_SD_WE  <= '0';
						NEXT_SD_DQM <= "0000";
						NEXT_SD_MBA <= "00";
						NEXT_SD_MA  <= "000000000000";
						NEXT_SD_CRD <= '0';
						NEXT_SD_CRA <= '0';
						NEXT_SD_CWD <= '0';
						NEXT_SD_CWA <= '0';
						NEXT_SD_PRD <= '1';
						NEXT_SD_PWD <= '0';
						NEXT_SD_REF <= '0';
						NEXT_COUNT  <= COUNT + 1;
						NEXT_STATE  <= STATE;
					when 3 to 9 =>			-- NOP
						NEXT_SD_CS  <= '1';
						NEXT_SD_RAS <= '0';
						NEXT_SD_CAS <= '0';
						NEXT_SD_WE  <= '0';
						NEXT_SD_DQM <= "0000";
						NEXT_SD_MBA <= "00";
						NEXT_SD_MA  <= "000000000000";
						NEXT_SD_CRD <= '0';
						NEXT_SD_CRA <= '0';
						NEXT_SD_CWD <= '0';
						NEXT_SD_CWA <= '0';
						NEXT_SD_PRD <= '0';
						NEXT_SD_PWD <= '0';
						NEXT_SD_REF <= '0';
						NEXT_COUNT  <= COUNT + 1;
						NEXT_STATE  <= STATE;
					when others =>			-- NOP
						NEXT_SD_CS  <= '1';
						NEXT_SD_RAS <= '0';
						NEXT_SD_CAS <= '0';
						NEXT_SD_WE  <= '0';
						NEXT_SD_DQM <= "0000";
						NEXT_SD_MBA <= "00";
						NEXT_SD_MA  <= "000000000000";
						NEXT_SD_CRD <= '0';
						NEXT_SD_CRA <= '0';
						NEXT_SD_CWD <= '0';
						NEXT_SD_CWA <= '0';
						NEXT_SD_PRD <= '0';
						NEXT_SD_PWD <= '0';
						NEXT_SD_REF <= '0';
						NEXT_COUNT  <= 0;
						NEXT_STATE  <= STBY;
				end case;
			when PWRITE =>
				DBG_OUT <= "0100";			-- DEBUG(-*--)
				case COUNT is
					when 0 =>				-- ACT
						NEXT_SD_CS  <= '1';
						NEXT_SD_RAS <= '1';
						NEXT_SD_CAS <= '0';
						NEXT_SD_WE  <= '0';
						NEXT_SD_DQM <= "0000";
						NEXT_SD_MBA <= CPUADDR(23 downto 22);
						NEXT_SD_MA  <= CPUADDR(21 downto 10);
						NEXT_SD_CRD <= '0';
						NEXT_SD_CRA <= '0';
						NEXT_SD_CWD <= '0';
						NEXT_SD_CWA <= '0';
						NEXT_SD_PRD <= '0';
						NEXT_SD_PWD <= '1';
						NEXT_SD_REF <= '0';
						NEXT_COUNT  <= COUNT + 1;
						NEXT_STATE  <= STATE;
					when 1 =>				-- WRITE(AUTO PRE)
						NEXT_SD_CS  <= '1';
						NEXT_SD_RAS <= '0';
						NEXT_SD_CAS <= '1';
						NEXT_SD_WE  <= '1';
						if (CPUADDR(1 downto 0) = "00") then
							NEXT_SD_DQM <= "1110";
						elsif (CPUADDR(1 downto 0) = "01") then
							NEXT_SD_DQM <= "1101";
						elsif (CPUADDR(1 downto 0) = "10") then
							NEXT_SD_DQM <= "1011";
						else
							NEXT_SD_DQM <= "0111";
						end if;
						NEXT_SD_MBA <= CPUADDR(23 downto 22);
						NEXT_SD_MA  <= "0100" & CPUADDR(9 downto  2);
						NEXT_SD_CRD <= '0';
						NEXT_SD_CRA <= '0';
						NEXT_SD_CWD <= '0';
						NEXT_SD_CWA <= '0';
						NEXT_SD_PRD <= '0';
						NEXT_SD_PWD <= '0';
						NEXT_SD_REF <= '0';
						NEXT_COUNT  <= COUNT + 1;
						NEXT_STATE  <= STATE;
					when 2 to 8 =>			-- NOP
						NEXT_SD_CS  <= '1';
						NEXT_SD_RAS <= '0';
						NEXT_SD_CAS <= '0';
						NEXT_SD_WE  <= '0';
						NEXT_SD_DQM <= "1111";
						NEXT_SD_MBA <= "00";
						NEXT_SD_MA  <= "000000000000";
						NEXT_SD_CRD <= '0';
						NEXT_SD_CRA <= '0';
						NEXT_SD_CWD <= '0';
						NEXT_SD_CWA <= '0';
						NEXT_SD_PRD <= '0';
						NEXT_SD_PWD <= '0';
						NEXT_SD_REF <= '0';
						NEXT_COUNT  <= COUNT + 1;
						NEXT_STATE  <= STATE;
					when 9 =>				-- NOP
						NEXT_SD_CS  <= '1';
						NEXT_SD_RAS <= '0';
						NEXT_SD_CAS <= '0';
						NEXT_SD_WE  <= '0';
						NEXT_SD_DQM <= "0000";
						NEXT_SD_MBA <= "00";
						NEXT_SD_MA  <= "000000000000";
						NEXT_SD_CRD <= '0';
						NEXT_SD_CRA <= '0';
						NEXT_SD_CWD <= '0';
						NEXT_SD_CWA <= '0';
						NEXT_SD_PRD <= '0';
						NEXT_SD_PWD <= '0';
						NEXT_SD_REF <= '0';
						NEXT_COUNT  <= COUNT + 1;
						NEXT_STATE  <= STATE;
					when others =>			-- NOP
						NEXT_SD_CS  <= '1';
						NEXT_SD_RAS <= '0';
						NEXT_SD_CAS <= '0';
						NEXT_SD_WE  <= '0';
						NEXT_SD_DQM <= "0000";
						NEXT_SD_MBA <= "00";
						NEXT_SD_MA  <= "000000000000";
						NEXT_SD_CRD <= '0';
						NEXT_SD_CRA <= '0';
						NEXT_SD_CWD <= '0';
						NEXT_SD_CWA <= '0';
						NEXT_SD_PRD <= '0';
						NEXT_SD_PWD <= '0';
						NEXT_SD_REF <= '0';
						NEXT_COUNT  <= 0;
						NEXT_STATE  <= STBY;
				end case;
			when REF =>
				DBG_OUT <= "1000";			-- DEBUG(*---)
				case COUNT is
					when 0 =>				-- REF
						NEXT_SD_CS  <= '1';
						NEXT_SD_RAS <= '1';
						NEXT_SD_CAS <= '1';
						NEXT_SD_WE  <= '0';
						NEXT_SD_DQM <= "0000";
						NEXT_SD_MBA <= "00";
						NEXT_SD_MA  <= "000000000000";
						NEXT_SD_CRD <= '0';
						NEXT_SD_CRA <= '0';
						NEXT_SD_CWD <= '0';
						NEXT_SD_CWA <= '0';
						NEXT_SD_PRD <= '0';
						NEXT_SD_PWD <= '0';
						NEXT_SD_REF <= '1';
						NEXT_COUNT  <= COUNT + 1;
						NEXT_STATE  <= STATE;
					when others =>			-- NOP
						NEXT_SD_CS  <= '1';
						NEXT_SD_RAS <= '0';
						NEXT_SD_CAS <= '0';
						NEXT_SD_WE  <= '0';
						NEXT_SD_DQM <= "0000";
						NEXT_SD_MBA <= "00";
						NEXT_SD_MA  <= "000000000000";
						NEXT_SD_CRD <= '0';
						NEXT_SD_CRA <= '0';
						NEXT_SD_CWD <= '0';
						NEXT_SD_CWA <= '0';
						NEXT_SD_PRD <= '0';
						NEXT_SD_PWD <= '0';
						NEXT_SD_REF <= '0';
						NEXT_COUNT  <= 0;
						NEXT_STATE  <= STBY;
				end case;
		end case;
	end process;

	SD_CKE <= INITREQ;

	process (CLK) begin
		if (CLK'event and CLK = '1') then
			SD_PWD_D <= SD_PWD;
		end if;
	end process;

	process (SD_PWD_D, CPUWDATA, CAMBUFRD) begin
		if (SD_PWD_D = '1') then
			SD_MD_o <= CPUWDATA & CPUWDATA & CPUWDATA & CPUWDATA;
		else
			SD_MD_o <= CAMBUFRD;
		end if;
	end process;

	process (CLK) begin
		if (CLK'event and CLK = '1') then
			SD_MD_t <= SD_CWD or SD_PWD;
		end if;
	end process;

end RTL;
