
------------------------------------------------------------------------------
-- Title       : processor.vhd
-- Description : vZbT
-- Author      : KAWAMOTO Yasuhisa
-- MAIL        : kawamoto@devdrv.co.jp
-- Date        : 11/07/2007
------------------------------------------------------------------------------

library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;

--               -----------------------------------------------
-- 00 LED       |  -  |  -  |  -  |  -  |  3  |  2  |  1  |  0  |
--               -----------------------------------------------
-- 01 SW        | NMI |  -  |  -  |  -  |  3  |  2  |  1  |  0  |
--               -----------------------------------------------
--              |                       |                       |
--              |                       |                       |
--              |                       |                       |
--               -----------------------------------------------
-- 10 UART_DATA |                      DATA                     |
--               -----------------------------------------------
-- 11 UART_CTRL | RXR | RXP | RXH | RXF | TXR |  -  | TXH | TXF |
--               -----------------------------------------------
--              |                       |                       |
--              |                       |                       |
--              |                       |                       |
--               -----------------------------------------------
-- 20 I2C_DATA  |                      DATA                     |
--               -----------------------------------------------
-- 21 I2C_CTRL  | STA | STO | ACK |  -  |  -  |  -  |  -  | DTP |
--               -----------------------------------------------
--              |                       |                       |
--              |                       |                       |
--              |                       |                       |
--               -----------------------------------------------
-- 30 CAM_CTRL  |  -  |  -  |  -  |  -  |  -  |  -  |  -  | CAM |
--               -----------------------------------------------
--              |                       |                       |
--              |                       |                       |
--              |                       |                       |
--               -----------------------------------------------
-- 40 MEM_ADDR0 |                      ADDR0                    |
--               -----------------------------------------------
-- 41 MEM_ADDR1 |                      ADDR1                    |
--               -----------------------------------------------
-- 42 MEM_ADDR2 |                      ADDR2                    |
--               -----------------------------------------------
-- 43 MEM_DATA  |                      DATA                     |
--               -----------------------------------------------
-- 44 MEM_CTRL  |  -  |  -  |  -  |  -  |  -  |  -  | MRD | MWR |
--               -----------------------------------------------
--              |                       |                       |
--              |                       |                       |
--              |                       |                       |
--               -----------------------------------------------
-- 50 DCT_DATA  |                      DATA                     |
--               -----------------------------------------------
-- 51 DCT_CTRL  |  -  |  -  |  -  |  -  |  -  |  -  |  -  | DCT |
--               -----------------------------------------------
--              |                       |                       |
--              |                       |                       |
--              |                       |                       |
--               -----------------------------------------------

entity PROCESSOR is
	port (
		RST : in std_logic;
		CLK : in std_logic;
		
		MEM_ADDR     : out std_logic_vector(23 downto 0);
		MEM_WDATA    : out std_logic_vector(7 downto 0);
		MEM_WDATASTB : out std_logic;
		MEM_WDATABSY : in std_logic;
		MEM_RDATA    : in std_logic_vector(7 downto 0);
		MEM_RDATASTB : out std_logic;
		MEM_RDATABSY : in std_logic;
		
		CAM_START : out std_logic;
		CAM_BUSY  : in std_logic;
		
		SERIAL_IN  : in std_logic;
		SERIAL_OUT : out std_logic;
		
		SCL_IN  : in std_logic;
		SCL_OUT : out std_logic;
		SDA_IN  : in std_logic;
		SDA_OUT : out std_logic;
		
		LED : out std_logic_vector(3 downto 0);
		SW  : in std_logic_vector(3 downto 0);
		NMI : in std_logic
	);
end PROCESSOR;

architecture RTL of PROCESSOR is

	component EMBEDDED_KCPSM3
	port(
		PORT_ID       : out std_logic_vector(7 downto 0);
		WRITE_STROBE  : out std_logic;
		READ_STROBE   : out std_logic;
		OUT_PORT      : out std_logic_vector(7 downto 0);
		IN_PORT       : in std_logic_vector(7 downto 0);
		INTERRUPT     : in std_logic;
		INTERRUPT_ACK : out std_logic;
		RESET         : in std_logic;
		CLK           : in std_logic
	);
	end component;

	component UART_TX
	port(
		DATA_IN          : in std_logic_vector(7 downto 0);
		WRITE_BUFFER     : in std_logic;
		RESET_BUFFER     : in std_logic;
		EN_16_X_BAUD     : in std_logic;
		SERIAL_OUT       : out std_logic;
		BUFFER_FULL      : out std_logic;
		BUFFER_HALF_FULL : out std_logic;
		clk              : in std_logic
	);
	end component;

	component UART_RX
	port(
		SERIAL_IN           : in std_logic;
		DATA_OUT            : out std_logic_vector(7 downto 0);
		READ_BUFFER         : in std_logic;
		RESET_BUFFER        : in std_logic;
		EN_16_X_BAUD        : in std_logic;
		BUFFER_DATA_PRESENT : out std_logic;
		BUFFER_FULL         : out std_logic;
		BUFFER_HALF_FULL    : out std_logic;
		CLK                 : in std_logic
	);
	end component;

	component I2C
	port(
		RST          : in std_logic;
		CLK          : in std_logic;
		EN_5_X_BAUD  : in std_logic;
		SCL_OUT      : out std_logic;
		SCL_IN       : in std_logic;
		SDA_OUT      : out std_logic;
		SDA_IN       : in std_logic;
		DATA_OUT     : out std_logic_vector(7 downto 0);
		ACK_OUT      : out std_logic;
		DATA_PRESENT : out std_logic;
		DATA_IN      : in std_logic_vector(7 downto 0);
		ACK_IN       : in std_logic;
		WRITE_STROBE : in std_logic;
		START        : in std_logic;
		STOP         : in std_logic
	);
	end component;

	component DCT
	port(
		RST          : in std_logic;
		CLK          : in std_logic;
		DATA_IN      : in std_logic_vector(7 downto 0);
		WRITE_STROBE : in std_logic;
		DATA_OUT     : out std_logic_vector(7 downto 0);
		READ_STROBE  : in std_logic;
		DCT_STROBE   : in std_logic;
		DCT_BUSY     : out std_logic
	);
	end component;

	signal PORT_ID      : std_logic_vector(7 downto 0);
	signal WRITE_STROBE : std_logic;
	signal READ_STROBE  : std_logic;
	signal OUT_PORT     : std_logic_vector(7 downto 0);
	signal IN_PORT      : std_logic_vector(7 downto 0);
	
	signal CS_WRITE_STROBE : std_logic_vector(255 downto 0);
	signal CS_READ_STROBE  : std_logic_vector(255 downto 0);

	type DEF_CS_IN_PORT is array (0 to 255) of std_logic_vector(7 downto 0);
	signal CS_IN_PORT : DEF_CS_IN_PORT;

	signal LED_OUT : std_logic_vector(3 downto 0);

	signal UART_CTRL_RXR : std_logic;
	signal UART_CTRL_RXP : std_logic;
	signal UART_CTRL_RXH : std_logic;
	signal UART_CTRL_RXF : std_logic;
	signal UART_CTRL_TXR : std_logic;
	signal UART_CTRL_TXH : std_logic;
	signal UART_CTRL_TXF : std_logic;

	signal UART_BAUD_CNT     : integer range 0 to 26;
	signal UART_EN_16_X_BAUD : std_logic;

	signal I2C_CTRL_STA     : std_logic;
	signal I2C_CTRL_STO     : std_logic;
	signal I2C_CTRL_ACK_IN  : std_logic;
	signal I2C_CTRL_ACK_OUT : std_logic;
	signal I2C_CTRL_DTP     : std_logic;

	signal I2C_BAUD_CNT    : integer range 0 to 30;
	signal I2C_EN_5_X_BAUD : std_logic;

	signal MEM_ADDR_OUT : std_logic_vector(23 downto 0);

	signal DCT_STROBE : std_logic;
	signal DCT_BUSY   : std_logic;

	signal NMI_S         : std_logic_vector(2 downto 1);
	signal INTERRUPT     : std_logic;
	signal INTERRUPT_ACK : std_logic;

begin

	U_EMBEDDED_KCPSM3 : EMBEDDED_KCPSM3
	port map(
		PORT_ID       => PORT_ID,
		WRITE_STROBE  => WRITE_STROBE,
		READ_STROBE   => READ_STROBE,
		OUT_PORT      => OUT_PORT,
		IN_PORT       => IN_PORT,
		INTERRUPT     => INTERRUPT,
		INTERRUPT_ACK => INTERRUPT_ACK,
		RESET         => RST,
		CLK           => CLK
	);

	process (CLK, RST) begin
		if (RST = '1') then
			NMI_S     <= (others => '0');
			INTERRUPT <= '0';
		elsif (CLK'event and CLK = '1') then
			NMI_S(1) <= NMI;
			NMI_S(2) <= NMI_S(1);
			if (NMI_S(1) = '1' and NMI_S(2) = '0') then
				INTERRUPT <= '1';
			elsif (INTERRUPT_ACK = '1') then
				INTERRUPT <= '0';
			end if;
		end if;
	end process;

	process (PORT_ID, READ_STROBE) begin
		for I in 0 to 255 loop
			if (PORT_ID = I and READ_STROBE = '1') then CS_READ_STROBE(I) <= '1'; else CS_READ_STROBE(I) <= '0'; end if;
		end loop;
	end process;

	process (PORT_ID, WRITE_STROBE) begin
		for I in 0 to 255 loop
			if (PORT_ID = I and WRITE_STROBE = '1') then CS_WRITE_STROBE(I) <= '1'; else CS_WRITE_STROBE(I) <= '0'; end if;
		end loop;
	end process;

	process (PORT_ID, CS_IN_PORT) begin
		IN_PORT <= CS_IN_PORT(conv_integer(PORT_ID));
	end process;

	process (CLK, RST) begin
		if (RST = '1') then
			LED_OUT <= (others => '0');
		elsif (CLK'event and CLK = '1') then
			if (CS_WRITE_STROBE(16#00#) = '1') then
				LED_OUT <= OUT_PORT(3 downto 0);
			end if;
		end if;
	end process;

	LED <= LED_OUT;

	CS_IN_PORT(16#00#) <= "0000" & LED_OUT;

	CS_IN_PORT(16#01#) <= NMI & "000" & SW;

	process (CLK, RST) begin
		if (RST = '1') then
			UART_BAUD_CNT     <= 0;
			UART_EN_16_X_BAUD <= '0';
		elsif (CLK'event and CLK = '1') then
			if (UART_BAUD_CNT = 25) then
				UART_BAUD_CNT     <= 0;
				UART_EN_16_X_BAUD <= '1';
			else
				UART_BAUD_CNT     <= UART_BAUD_CNT + 1;
				UART_EN_16_X_BAUD <= '0';
			end if;
		end if;
	end process;

	U_UART_TX : UART_TX
	port map(
		DATA_IN          => OUT_PORT,
		WRITE_BUFFER     => CS_WRITE_STROBE(16#10#),
		RESET_BUFFER     => UART_CTRL_TXR,
		EN_16_X_BAUD     => UART_EN_16_X_BAUD,
		SERIAL_OUT       => SERIAL_OUT,
		BUFFER_FULL      => UART_CTRL_TXF,
		BUFFER_HALF_FULL => UART_CTRL_TXH,
		clk              => CLK
	);

	U_UART_RX : UART_RX
	port map(
		SERIAL_IN           => SERIAL_IN,
		DATA_OUT            => CS_IN_PORT(16),
		READ_BUFFER         => CS_READ_STROBE(16#10#),
		RESET_BUFFER        => UART_CTRL_RXR,
		EN_16_X_BAUD        => UART_EN_16_X_BAUD,
		BUFFER_DATA_PRESENT => UART_CTRL_RXP,
		BUFFER_FULL         => UART_CTRL_RXF,
		BUFFER_HALF_FULL    => UART_CTRL_RXH,
		CLK                 => CLK
	);

	process (CLK, RST) begin
		if (RST = '1') then
			UART_CTRL_RXR <= '1';
			UART_CTRL_TXR <= '1';
		elsif (CLK'event and CLK = '1') then
			if (CS_WRITE_STROBE(16#11#) = '1') then
				UART_CTRL_RXR <= OUT_PORT(7);
				UART_CTRL_TXR <= OUT_PORT(4);
			end if;
		end if;
	end process;

	CS_IN_PORT(16#11#) <= UART_CTRL_RXR
		& UART_CTRL_RXP
		& UART_CTRL_RXH
		& UART_CTRL_RXF
		& UART_CTRL_TXR
		& '0'
		& UART_CTRL_TXH
		& UART_CTRL_TXF;

	process (CLK, RST) begin
		if (RST = '1') then
			I2C_BAUD_CNT    <= 0;
			I2C_EN_5_X_BAUD <= '0';
		elsif (CLK'event and CLK = '1') then
			if (I2C_BAUD_CNT = 29) then
				I2C_BAUD_CNT    <= 0;
				I2C_EN_5_X_BAUD <= '1';
			else
				I2C_BAUD_CNT    <= I2C_BAUD_CNT + 1;
				I2C_EN_5_X_BAUD <= '0';
			end if;
		end if;
	end process;

	U_I2C : I2C
	port map(
		RST          => RST,
		CLK          => CLK,
		EN_5_X_BAUD  => I2C_EN_5_X_BAUD,
		SCL_OUT      => SCL_OUT,
		SCL_IN       => SCL_IN,
		SDA_OUT      => SDA_OUT,
		SDA_IN       => SDA_IN,
		DATA_OUT     => CS_IN_PORT(16#20#),
		ACK_OUT      => I2C_CTRL_ACK_IN,
		DATA_PRESENT => I2C_CTRL_DTP,
		DATA_IN      => OUT_PORT,
		ACK_IN       => I2C_CTRL_ACK_OUT,
		WRITE_STROBE => CS_WRITE_STROBE(16#20#),
		START        => I2C_CTRL_STA,
		STOP         => I2C_CTRL_STO
	);

	process (CLK, RST) begin
		if (RST = '1') then
			I2C_CTRL_STA     <= '0';
			I2C_CTRL_STO     <= '0';
			I2C_CTRL_ACK_OUT <= '0';
		elsif (CLK'event and CLK = '1') then
			if (CS_WRITE_STROBE(16#21#) = '1') then
				I2C_CTRL_STA     <= OUT_PORT(7);
				I2C_CTRL_STO     <= OUT_PORT(6);
				I2C_CTRL_ACK_OUT <= OUT_PORT(5);
			end if;
		end if;
	end process;

	CS_IN_PORT(16#21#) <= I2C_CTRL_STA
		& I2C_CTRL_STO
		& I2C_CTRL_ACK_IN
		& "0000"
		& I2C_CTRL_DTP;

	process (CLK, RST) begin
		if (RST = '1') then
			CAM_START <= '0';
		elsif (CLK'event and CLK = '1') then
			if (CS_WRITE_STROBE(16#30#) = '1') then
				CAM_START <= OUT_PORT(0);
			end if;
		end if;
	end process;

	CS_IN_PORT(16#30#) <= "0000000" & CAM_BUSY;

	process (CLK, RST) begin
		if (RST = '1') then
			MEM_ADDR_OUT <= (others => '0');
		elsif (CLK'event and CLK = '1') then
			if (CS_WRITE_STROBE(16#40#) = '1') then
				MEM_ADDR_OUT(7 downto 0) <= OUT_PORT;
			end if;
			if (CS_WRITE_STROBE(16#41#) = '1') then
				MEM_ADDR_OUT(15 downto 8) <= OUT_PORT;
			end if;
			if (CS_WRITE_STROBE(16#42#) = '1') then
				MEM_ADDR_OUT(23 downto 16) <= OUT_PORT;
			end if;
		end if;
	end process;

	MEM_ADDR <= MEM_ADDR_OUT;

	CS_IN_PORT(16#40#) <= MEM_ADDR_OUT( 7 downto  0);
	CS_IN_PORT(16#41#) <= MEM_ADDR_OUT(15 downto  8);
	CS_IN_PORT(16#42#) <= MEM_ADDR_OUT(23 downto 16);

	process (CLK, RST) begin
		if (RST = '1') then
			MEM_WDATA <= (others => '0');
		elsif (CLK'event and CLK = '1') then
			if (CS_WRITE_STROBE(16#43#) = '1') then
				MEM_WDATA <= OUT_PORT;
			end if;
		end if;
	end process;

	CS_IN_PORT(16#43#) <= MEM_RDATA;

	MEM_RDATASTB <= CS_WRITE_STROBE(16#44#) and OUT_PORT(1);
	MEM_WDATASTB <= CS_WRITE_STROBE(16#44#) and OUT_PORT(0);

	CS_IN_PORT(16#44#) <= "000000"
		& MEM_RDATABSY
		& MEM_WDATABSY;

	U_DCT : DCT
	port map(
		RST          => RST,
		CLK          => CLK,
		DATA_IN      => OUT_PORT,
		WRITE_STROBE => CS_WRITE_STROBE(16#50#),
		DATA_OUT     => CS_IN_PORT(16#50#),
		READ_STROBE  => CS_READ_STROBE(16#50#),
		DCT_STROBE   => DCT_STROBE,
		DCT_BUSY     => DCT_BUSY
	);

	DCT_STROBE <= CS_WRITE_STROBE(16#51#) and OUT_PORT(0);

	CS_IN_PORT(16#51#) <= "0000000" & DCT_BUSY;

end RTL;
