
------------------------------------------------------------------------------
-- Title       : dct.vhd
-- Description : DCT ϊ
-- Author      : KAWAMOTO Yasuhisa
-- Mail        : kawamoto@devdrv.co.jp
-- Date        : 09/06/2007
------------------------------------------------------------------------------

library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_signed.all;

entity DCT is
	port(
		RST          : in std_logic;
		CLK          : in std_logic;
		DATA_IN      : in std_logic_vector(7 downto 0);
		WRITE_STROBE : in std_logic;
		DATA_OUT     : out std_logic_vector(7 downto 0);
		READ_STROBE  : in std_logic;
		DCT_STROBE   : in std_logic;
		DCT_BUSY     : out std_logic
	);
end DCT;

architecture RTL of DCT is
	type DEF_DCTMATRIX is array (0 to 63) of integer range -128 to 127;
	constant DCTMATRIX : DEF_DCTMATRIX :=
	(
		 32,  32,  32,  32,  32,  32,  32,  32,
		 44,  38,  25,   9,  -9, -25, -38, -44,
		 42,  17, -17, -42, -42, -17,  17,  42,
		 38,  -9, -44, -25,  25,  44,   9, -38,
		 32, -32, -32,  32,  32, -32, -32,  32,
		 25, -44,   9,  38, -38,  -9,  44, -25,
		 17, -42,  42, -17, -17,  42, -42,  17,
		  9, -25,  38, -44,  44, -38,  25,  -9
	);

	type DEF_BUFF is array (0 to 7) of std_logic_vector(7 downto 0);
	signal IN_BUFF  : DEF_BUFF;
	signal OUT_BUFF : DEF_BUFF;

	signal X_COUNT : integer range 0 to 7;
	signal Y_COUNT : integer range 0 to 7;

	signal DCT_MAC  : std_logic_vector(15 downto 0);
	signal DCT_TEMP : std_logic_vector(15 downto 0);
	signal OUT_TEMP : std_logic_vector(8 downto 0);

	signal DCT_BUSY_OUT : std_logic;

begin

	process(CLK) begin
		if (CLK'event and CLK = '1') then
			if (WRITE_STROBE = '1') then
				IN_BUFF(7) <= DATA_IN;
				IN_BUFF(6) <= IN_BUFF(7);
				IN_BUFF(5) <= IN_BUFF(6);
				IN_BUFF(4) <= IN_BUFF(5);
				IN_BUFF(3) <= IN_BUFF(4);
				IN_BUFF(2) <= IN_BUFF(3);
				IN_BUFF(1) <= IN_BUFF(2);
				IN_BUFF(0) <= IN_BUFF(1);
			end if;
		end if;
	end process;

	process(X_COUNT, Y_COUNT, IN_BUFF, DCT_MAC) begin
		if (X_COUNT = 0) then
			DCT_TEMP <= IN_BUFF(X_COUNT)
					* CONV_STD_LOGIC_VECTOR(DCTMATRIX(X_COUNT + Y_COUNT * 8), 8);
		else
			DCT_TEMP <= DCT_MAC + IN_BUFF(X_COUNT)
					* CONV_STD_LOGIC_VECTOR(DCTMATRIX(X_COUNT + Y_COUNT * 8), 8);
		end if;
	end process;

	process(CLK) begin
		if (CLK'event and CLK = '1') then
			if (DCT_BUSY_OUT = '1') then
				DCT_MAC <= DCT_TEMP;
			end if;
		end if;
	end process;

	process(DCT_TEMP) begin
		OUT_TEMP <= DCT_TEMP(15 downto 7) + ("00000000" & DCT_TEMP(15));
	end process;

	process(CLK) begin
		if (CLK'event and CLK = '1') then
			if (READ_STROBE = '1') then
				OUT_BUFF(6) <= OUT_BUFF(7);
				OUT_BUFF(5) <= OUT_BUFF(6);
				OUT_BUFF(4) <= OUT_BUFF(5);
				OUT_BUFF(3) <= OUT_BUFF(4);
				OUT_BUFF(2) <= OUT_BUFF(3);
				OUT_BUFF(1) <= OUT_BUFF(2);
				OUT_BUFF(0) <= OUT_BUFF(1);
			elsif (DCT_BUSY_OUT = '1') then
				OUT_BUFF(Y_COUNT) <= OUT_TEMP(8 downto 1);
			end if;
		end if;
	end process;

	DATA_OUT <= OUT_BUFF(0);

	process (CLK, RST) begin
		if (RST = '1') then
			X_COUNT <= 0;
		elsif (CLK'event and CLK = '1') then
			if (DCT_BUSY_OUT = '1') then
				if (X_COUNT = 7) then
					X_COUNT <= 0;
				else
					X_COUNT <= X_COUNT + 1;
				end if;
			end if;
		end if;
	end process;

	process (CLK, RST) begin
		if (RST = '1') then
			Y_COUNT <= 0;
		elsif (CLK'event and CLK = '1') then
			if ((DCT_BUSY_OUT = '1') and (X_COUNT = 7)) then
				if (Y_COUNT = 7) then
					Y_COUNT <= 0;
				else
					Y_COUNT <= Y_COUNT + 1;
				end if;
			end if;
		end if;
	end process;

	process (CLK, RST) begin
		if (RST = '1') then
			DCT_BUSY_OUT <= '0';
		elsif (CLK'event and CLK = '1') then
			if ((X_COUNT = 7) and (Y_COUNT = 7)) then
				DCT_BUSY_OUT <= '0';
			elsif (DCT_STROBE = '1') then
				DCT_BUSY_OUT <= '1';
			end if;
		end if;
	end process;

	DCT_BUSY <= DCT_BUSY_OUT;

end RTL;
