/*
 * USB EZCam Camera driver by Device Drivers Limited (http://www.devdrv.co.jp/)
 *     Modified from IBM USB NB Camera driver
 *     9/26/2003
 *     4/11/2005 ported to linux 2.6
 */

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/module.h>
#include <linux/init.h>

#include "usbvideo.h"

#define	EZCAM_VENDOR_ID	        0x0547
#define	EZCAM_PRODUCT_ID	0x1002

#define MAX_CAMERAS		4	/* How many devices we allow to connect */

/*
 * This structure lives in uvd_t->user field.
 */
typedef struct {
	int initialized;	/* Had we already sent init sequence? */
	int camera_model;	/* What type of IBM camera we got? */
        int has_hdr;
} ezcam_t;
#define	EZCAM_T(uvd)	((ezcam_t *)((uvd)->user_data))

static struct usbvideo *cams = NULL;

static int debug = 5;

static int flags = 0; /* FLAGS_DISPLAY_HINTS | FLAGS_OVERLAY_STATS; */

static const int min_canvasWidth  = 8;
static const int min_canvasHeight = 4;

#define FRAMERATE_MIN	0
#define FRAMERATE_MAX	6
static int framerate = -1;

/*
 * Here we define several initialization variables. They may
 * be used to automatically set color, hue, brightness and
 * contrast to desired values. This is particularly useful in
 * case of webcams (which have no controls and no on-screen
 * output) and also when a client V4L software is used that
 * does not have some of those controls. In any case it's
 * good to have startup values as options.
 *
 * These values are all in [0..255] range. This simplifies
 * operation. Note that actual values of V4L variables may
 * be scaled up (as much as << 8). User can see that only
 * on overlay output, however, or through a V4L client.
 */
static int init_brightness = 128;
static int init_contrast = 192;
static int init_color = 128;
static int init_hue = 128;

module_param(debug, int, S_IRUGO | S_IWUSR);
MODULE_PARM_DESC(debug, "Debug level: 0-9 (default=0)");

/*
 * ezcam_ProcessIsocData()
 *
 * Generic routine to parse the ring queue data. It employs either
 * ezcam_find_header() or ezcam_parse_lines() to do most
 * of work.
 *
 * 02-Nov-2000 First (mostly dummy) version.
 * 06-Nov-2000 Rewrote to dump all data into frame.
 */

static void ezcam_ProcessIsocData(struct uvd *uvd, struct usbvideo_frame *frame)
{
        int n, i, c;
        int m = -1;
        unsigned char *temp;
        unsigned char second;
        unsigned short work;
        unsigned char *p;
        unsigned char *q;
        static unsigned char first = 0xFF;

	assert(uvd != NULL);
	assert(frame != NULL);

	/* Try to move data from queue into frame buffer */
	n = RingQueue_GetLength(&uvd->dp);
	if (n > 0) {
		if (frame->seqRead_Length == 0) {
			do {
				c = RingQueue_Dequeue(
					&uvd->dp,
					(void *) &first,
					1);
				if (first & 0x80) {
					/* got first data */
					first &= 0x7F;
					break;
				}
			} while (c == 1);
			n = RingQueue_GetLength(&uvd->dp);
			if (c < 1 || n <= 0)
				return;
		}
		m = uvd->max_frame_size - frame->seqRead_Length;
		if (n > m)
			n = m;

                temp = (unsigned char *)kmalloc(n, GFP_KERNEL);
                if (temp == NULL) {
                        err("ProcessIsoData kmalloc error = %d", n);
                        return;
                }

		/* Now move that much data into frame buffer */
		RingQueue_Dequeue(
			&uvd->dp,
			(void *) temp,
			n);

                /* convert Number of Colors */
                p = frame->data + frame->seqRead_Length;
		q = temp;
		c = 0;
                for(i = 0; i < n; i++) {
			if (first != 0xFF) {
				second = *q++;
				work = (second & 0x7F) | ((first & 0x7F) << 7);
				*p++ = (work & 0x000F) << 4; /* blue */
				*p++ = (work & 0x01F0) >> 1; /* green */
				*p++ = (work & 0x3E00) >> 6; /* red */
				c += 3;
				first = 0xFF;
			}
			else {
				first = *q++;
			}
                }
		frame->seqRead_Length += c;
		kfree(temp);
	}
	/* See if we filled the frame */
	if (frame->seqRead_Length >= uvd->max_frame_size) {
		frame->frameState = FrameState_Done;
		uvd->curframe = -1;
		uvd->stats.frame_num++;
	}
}

/*
 * ezcam_calculate_fps()
 */
static int ezcam_calculate_fps(struct uvd *uvd)
{
	return 3 + framerate*4 + framerate/2;
}

/*
 * ezcam_adjust_picture()
 *
 * This procedure gets called from V4L interface to update picture settings.
 * Here we change brightness and contrast.
 */
static void ezcam_adjust_picture(struct uvd *uvd)
{
}

/*
 * ezcam_video_stop()
 *
 * This code tells camera to stop streaming. The interface remains
 * configured and bandwidth - claimed.
 */
static void ezcam_video_stop(struct uvd *uvd)
{
	if (debug >= 1)
		info("ezcam_video_start");
}

static void ezcam_video_start(struct uvd *uvd)
{
	if (debug >= 1)
		info("ezcam_video_start");
}

static int ezcam_resetPipe(struct uvd *uvd)
{
	usb_clear_halt(uvd->dev, uvd->video_endp);
	return 0;
}

/*
 * Return negative code on failure, 0 on success.
 */
static int ezcam_setup_on_open(struct uvd *uvd)
{
	int setup_ok = 0; /* Success by default */
	/* Send init sequence only once, it's large! */
	if (!EZCAM_T(uvd)->initialized) {
		ezcam_resetPipe(uvd);
		EZCAM_T(uvd)->initialized = (setup_ok != 0);
	}
	return setup_ok;
}

static void ezcam_configure_video(struct uvd *uvd)
{
	if (uvd == NULL)
		return;
	memset(&uvd->vpic, 0, sizeof(uvd->vpic));
	memset(&uvd->vpic_old, 0x55, sizeof(uvd->vpic_old));

	uvd->vpic.colour = init_color << 8;
	uvd->vpic.hue = init_hue << 8;
	uvd->vpic.brightness = init_brightness << 8;
	uvd->vpic.contrast = init_contrast << 8;
	uvd->vpic.whiteness = 105 << 8; /* This one isn't used */
	uvd->vpic.depth = 24;
	uvd->vpic.palette = VIDEO_PALETTE_RGB24;

	memset(&uvd->vcap, 0, sizeof(uvd->vcap));
	strcpy(uvd->vcap.name, "IBM Ultra Camera");
	uvd->vcap.type = VID_TYPE_CAPTURE;
	uvd->vcap.channels = 1;
	uvd->vcap.audios = 0;
	uvd->vcap.maxwidth = VIDEOSIZE_X(uvd->canvas);
	uvd->vcap.maxheight = VIDEOSIZE_Y(uvd->canvas);
	uvd->vcap.minwidth = min_canvasWidth;
	uvd->vcap.minheight = min_canvasHeight;

	memset(&uvd->vchan, 0, sizeof(uvd->vchan));
	uvd->vchan.flags = 0;
	uvd->vchan.tuners = 0;
	uvd->vchan.channel = 0;
	uvd->vchan.type = VIDEO_TYPE_CAMERA;
	strcpy(uvd->vchan.name, "Camera");
}

/*
 * ezcam_probe()
 *
 * This procedure queries device descriptor and accepts the interface
 * if it looks like our camera.
 *
 * History:
 * 12-Nov-2000 Reworked to comply with new probe() signature.
 * 23-Jan-2001 Added compatibility with 2.2.x kernels.
 * 11-Apl-2005 ported to linux 2.6
 */
static int ezcam_probe(struct usb_interface *intf, const struct usb_device_id *devid)
{
        struct usb_device *dev = interface_to_usbdev(intf);
	struct uvd *uvd = NULL;
	int ix, i=0, nas;
	int actInterface=-1, inactInterface=-1, maxPS=0;
	unsigned char video_ep = 0;

	if (debug >= 1)
		info("ezcam_probe(%p,%p)", dev, intf);

	/* We don't handle multi-config cameras */
	if (dev->descriptor.bNumConfigurations != 1)
		return -ENODEV;

	info("Device Drivers EZ camera found (rev. 0x%04x)",
	     le16_to_cpu(dev->descriptor.bcdDevice));

	/* Validate found interface: must have one ISO endpoint */
        nas = intf->num_altsetting;
	if (debug > 0) {
		info("Nunmber of alternate settings=%d.", nas);
	}
	/* Validate all alternate settings */
	for(ix = 0; ix < nas; ix++) {
		const struct usb_host_interface *interface;
		const struct usb_endpoint_descriptor *endpoint;

                interface = &intf->altsetting[ix];
                i = interface->desc.bAlternateSetting;
                if (interface->desc.bNumEndpoints != 1) {
                        err("Interface %d. has %u. endpoints!",
                            interface->desc.bInterfaceNumber,
                            (unsigned)(interface->desc.bNumEndpoints));
                        return -ENODEV;
		}
		endpoint = &interface->endpoint[0].desc;
		if (video_ep == 0)
			video_ep = endpoint->bEndpointAddress;
		else if (video_ep != endpoint->bEndpointAddress) {
			err("Alternate settings have different endpoint addresses!");
			return -ENODEV;
		}
		if ((endpoint->bmAttributes & 0x03) != 0x01) {
                        err("Interface %d. has non-ISO endpoint!",
                            interface->desc.bInterfaceNumber);
                        return -ENODEV;
		}
		if ((endpoint->bEndpointAddress & 0x80) == 0) {
                        err("Interface %d. has ISO OUT endpoint!",
                            interface->desc.bInterfaceNumber);
			return -ENODEV;
		}
                if (le16_to_cpu(endpoint->wMaxPacketSize) == 0) {
                        if (inactInterface < 0)
                                inactInterface = i;
                        else {
                                err("More than one inactive alt. setting!");
                                return -ENODEV;
                        }
		} else {
                        if (actInterface < 0) {
                                actInterface = i;
                                maxPS = le16_to_cpu(endpoint->wMaxPacketSize);
                                if (debug > 0)
                                        info("Active setting=%d. maxPS=%d.", i, maxPS);
                        } else {
                                /* Got another active alt. setting */
                                if (maxPS < le16_to_cpu(endpoint->wMaxPacketSize)) {
                                        /* This one is better! */
                                        actInterface = i;
                                        maxPS = le16_to_cpu(endpoint->wMaxPacketSize);
                                        if (debug > 0) {
                                                info("Even better ctive setting=%d. maxPS=%d.",
                                                     i, maxPS);
                                        }
                                }
                        }
		}
	}
	if (debug > 0) {
		info("maxPS = %d, actInterface = %d, inactInterface = %d",
		     maxPS, actInterface, inactInterface);
	}
	uvd = usbvideo_AllocateDevice(cams);
	if (uvd != NULL) {
		/* Here uvd is a fully allocated struct uvd object */
		uvd->flags = flags;
		uvd->debug = debug;
		uvd->dev = dev;
                uvd->iface = intf->altsetting->desc.bInterfaceNumber;
		/* uvd->ifaceAltInactive = inactInterface; */
		/* uvd->ifaceAltActive = actInterface; */
		uvd->video_endp = video_ep;
		uvd->iso_packet_len = maxPS;
		uvd->paletteBits = 1L << VIDEO_PALETTE_RGB24;
		uvd->defaultPalette = VIDEO_PALETTE_RGB24;
		uvd->canvas = VIDEOSIZE(160, 144);	/* FIXME */
		uvd->videosize = uvd->canvas; /* ezcam_size_to_videosize(size);*/

		/* Initialize ibmcam-specific data */
		assert(EZCAM_T(uvd) != NULL);
		EZCAM_T(uvd)->camera_model = 0; /* Not used yet */
		EZCAM_T(uvd)->initialized = 0;

		ezcam_configure_video(uvd);

		i = usbvideo_RegisterVideoDevice(uvd);
		if (i != 0) {
			err("usbvideo_RegisterVideoDevice() failed.");
			uvd = NULL;
		}
	}
        if (uvd) {
                usb_set_intfdata (intf, uvd);
                return 0;
        }
	return -EIO;
}

static struct usb_device_id id_table[] = {
        { USB_DEVICE(EZCAM_VENDOR_ID, EZCAM_PRODUCT_ID) },
        { }  /* Terminating entry */
};

/*
 * ezcam_init()
 *
 * This code is run to initialize the driver.
 */
static int __init ezcam_init(void)
{
	struct usbvideo_cb cbTbl;
	memset(&cbTbl, 0, sizeof(cbTbl));
	cbTbl.probe = ezcam_probe;
	cbTbl.setupOnOpen = ezcam_setup_on_open;
	cbTbl.videoStart = ezcam_video_start;
	cbTbl.videoStop = ezcam_video_stop;
	cbTbl.processData = ezcam_ProcessIsocData;
	cbTbl.postProcess = usbvideo_DeinterlaceFrame;
	cbTbl.adjustPicture = ezcam_adjust_picture;
	cbTbl.getFPS = ezcam_calculate_fps;
	return usbvideo_register(
		&cams,
		MAX_CAMERAS,
		sizeof(ezcam_t),
		"ezcam",
		&cbTbl,
		THIS_MODULE,
		id_table);
}

static void __exit ezcam_cleanup(void)
{
	usbvideo_Deregister(&cams);
}

MODULE_DEVICE_TABLE(usb, id_table);
MODULE_LICENSE("GPL");

module_init(ezcam_init);
module_exit(ezcam_cleanup);
