﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO; //MemoryStream
using Microsoft.Win32.SafeHandles;

namespace RX62N
{
    internal class RX62Function
    {
        private const String USB_GUID_STRING = "{94A23B10-6D99-4903-AD6E-4042D723E4AB}";
        private DeviceManagement myDeviceManagement = new DeviceManagement();
        private String myDevicePathName;
        private UsbDevice myUsbDevice = new UsbDevice();

        private delegate void ReadFromDeviceDelegate
            (Byte pipeID,
            UInt32 bufferLength,
            ref Byte[] buffer,
            ref UInt32 lengthTransferred,
            ref Boolean success);

        public RX62Function()
        {
            Boolean deviceFound;
            String devicePathName = "";
            Boolean success;

            try
            {
                System.Guid UsbGuid = new System.Guid(USB_GUID_STRING);
                deviceFound = myDeviceManagement.FindDeviceFromGuid
                    (UsbGuid,
                    ref devicePathName);

                if (!deviceFound)
                {
                    Console.WriteLine("Device GUID is not matched:" + USB_GUID_STRING);
                    return;
                }

                success = myUsbDevice.GetDeviceHandle(devicePathName);
                if (success)
                {
                    Console.WriteLine("Device detected:" + devicePathName);
                    myDevicePathName = devicePathName;
                }
                else
                {
                    Console.WriteLine("Device is not detected:" + devicePathName);
                    myUsbDevice.CloseDeviceHandle();
                    return;
                }
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                throw;
            }
        }

        ~RX62Function()
        {
            this.Close();
        }

        public void Close()
        {
            myUsbDevice.CloseDeviceHandle();
        }

        public int GetDipSW()
        {
            int sw;
            int count;
            byte[] buffer = new byte[1];

            count = myUsbDevice.Read(buffer, 1);
            sw = (int) buffer[0];
            return sw;
        }

        public bool SetLED(int pattern)
        {
            int count;
            byte[] buffer = new byte[1];
            
            buffer[0] = (byte) pattern;
            count = myUsbDevice.Write(buffer, 1);
            return (count == 1);
        }

        public bool GetIntreruptSW(byte[] buffer)
        {
            int count = myUsbDevice.InterruptIn(buffer, 1);
            return (count == 1);
        }
    }

    sealed internal partial class UsbDevice
    {
        internal struct devInfo
        {
            internal SafeFileHandle deviceHandle;
            internal FileStream usbDevice;
            internal BinaryReader reader;
            internal BinaryWriter writer;
        }
        internal devInfo myDevInfo = new devInfo();

        internal int Read(byte[] buffer, int count)
        {
            int bytes = -1;

            try
            {
                if (!(myDevInfo.deviceHandle.IsInvalid))
                {
                    bytes = myDevInfo.reader.Read(buffer, 0, count);
                }
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                throw;
            }
            return bytes;
        }

        internal int Write(byte[] buffer, int count)
        {
            int bytes = -1;

            try
            {
                if (!(myDevInfo.deviceHandle.IsInvalid))
                {
                    myDevInfo.writer.Write(buffer, 0, count);
                    myDevInfo.writer.Flush();
                    bytes = count;
                }
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                throw;
            }
            return bytes;
        }

        internal int InterruptIn(byte[] buffer, int count)
        {
            //UInt32 switchState;
            int index = 0;

            try
            {
                if (!(myDevInfo.deviceHandle.IsInvalid))
                {
                    bool result = FileIO.DeviceIoControl(
                        myDevInfo.deviceHandle,
                        FileIO.IOCTL_RX62N_GET_INTERRUPT_MESSAGE,
                        null,             // Ptr to InBuffer
                        0,                // Length of InBuffer
                        buffer,           // Ptr to OutBuffer
                        (uint) count,     // Length of OutBuffer
                        ref index,        // BytesReturned
                        0);               // Ptr to Overlapped structure

                    //switchState = buffer[0];
                    if (result)
                    {
                        index = 1;
                    }
                }
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                throw;
            }
            return index;
        }

        internal void CloseDeviceHandle()
        {
            try
            {
                if (!(myDevInfo.deviceHandle.IsInvalid))
                {
                    myDevInfo.usbDevice.Close();
                    myDevInfo.deviceHandle.Close();
                }
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                throw;
            }
        }

        internal Boolean GetDeviceHandle(String devicePathName)
        {
            myDevInfo.deviceHandle = FileIO.CreateFile(
                devicePathName,
                (FileIO.GENERIC_WRITE | FileIO.GENERIC_READ),
                FileIO.FILE_SHARE_READ | FileIO.FILE_SHARE_WRITE,
                IntPtr.Zero,
                FileIO.OPEN_EXISTING,
                FileIO.FILE_ATTRIBUTE_NORMAL,
                0);

            if (!(myDevInfo.deviceHandle.IsInvalid))
            {

                myDevInfo.usbDevice = new FileStream(myDevInfo.deviceHandle, FileAccess.ReadWrite);
                if (myDevInfo.usbDevice != null)
                {
                    myDevInfo.reader = new BinaryReader(myDevInfo.usbDevice);
                    myDevInfo.writer = new BinaryWriter(myDevInfo.usbDevice);
                }
                return true;
            }
            else
            {
                return false;
            }
        }
    }
}
